﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/guardduty/GuardDuty_EXPORTS.h>
#include <aws/guardduty/model/MalwareProtectionFindingsScanConfiguration.h>
#include <aws/guardduty/model/MalwareProtectionScanType.h>
#include <aws/guardduty/model/ScanCategory.h>
#include <aws/guardduty/model/Threat.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GuardDuty {
namespace Model {

/**
 * <p>Information about the malware scan that generated a GuardDuty
 * finding.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/MalwareScanDetails">AWS
 * API Reference</a></p>
 */
class MalwareScanDetails {
 public:
  AWS_GUARDDUTY_API MalwareScanDetails() = default;
  AWS_GUARDDUTY_API MalwareScanDetails(Aws::Utils::Json::JsonView jsonValue);
  AWS_GUARDDUTY_API MalwareScanDetails& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GUARDDUTY_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Information about the detected threats associated with the generated
   * GuardDuty finding.</p>
   */
  inline const Aws::Vector<Threat>& GetThreats() const { return m_threats; }
  inline bool ThreatsHasBeenSet() const { return m_threatsHasBeenSet; }
  template <typename ThreatsT = Aws::Vector<Threat>>
  void SetThreats(ThreatsT&& value) {
    m_threatsHasBeenSet = true;
    m_threats = std::forward<ThreatsT>(value);
  }
  template <typename ThreatsT = Aws::Vector<Threat>>
  MalwareScanDetails& WithThreats(ThreatsT&& value) {
    SetThreats(std::forward<ThreatsT>(value));
    return *this;
  }
  template <typename ThreatsT = Threat>
  MalwareScanDetails& AddThreats(ThreatsT&& value) {
    m_threatsHasBeenSet = true;
    m_threats.emplace_back(std::forward<ThreatsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for the malware scan.</p>
   */
  inline const Aws::String& GetScanId() const { return m_scanId; }
  inline bool ScanIdHasBeenSet() const { return m_scanIdHasBeenSet; }
  template <typename ScanIdT = Aws::String>
  void SetScanId(ScanIdT&& value) {
    m_scanIdHasBeenSet = true;
    m_scanId = std::forward<ScanIdT>(value);
  }
  template <typename ScanIdT = Aws::String>
  MalwareScanDetails& WithScanId(ScanIdT&& value) {
    SetScanId(std::forward<ScanIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of malware scan performed.</p>
   */
  inline MalwareProtectionScanType GetScanType() const { return m_scanType; }
  inline bool ScanTypeHasBeenSet() const { return m_scanTypeHasBeenSet; }
  inline void SetScanType(MalwareProtectionScanType value) {
    m_scanTypeHasBeenSet = true;
    m_scanType = value;
  }
  inline MalwareScanDetails& WithScanType(MalwareProtectionScanType value) {
    SetScanType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The category of the malware scan.</p>
   */
  inline ScanCategory GetScanCategory() const { return m_scanCategory; }
  inline bool ScanCategoryHasBeenSet() const { return m_scanCategoryHasBeenSet; }
  inline void SetScanCategory(ScanCategory value) {
    m_scanCategoryHasBeenSet = true;
    m_scanCategory = value;
  }
  inline MalwareScanDetails& WithScanCategory(ScanCategory value) {
    SetScanCategory(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration settings used for the malware scan.</p>
   */
  inline const MalwareProtectionFindingsScanConfiguration& GetScanConfiguration() const { return m_scanConfiguration; }
  inline bool ScanConfigurationHasBeenSet() const { return m_scanConfigurationHasBeenSet; }
  template <typename ScanConfigurationT = MalwareProtectionFindingsScanConfiguration>
  void SetScanConfiguration(ScanConfigurationT&& value) {
    m_scanConfigurationHasBeenSet = true;
    m_scanConfiguration = std::forward<ScanConfigurationT>(value);
  }
  template <typename ScanConfigurationT = MalwareProtectionFindingsScanConfiguration>
  MalwareScanDetails& WithScanConfiguration(ScanConfigurationT&& value) {
    SetScanConfiguration(std::forward<ScanConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of unique malware threats detected during the scan.</p>
   */
  inline int GetUniqueThreatCount() const { return m_uniqueThreatCount; }
  inline bool UniqueThreatCountHasBeenSet() const { return m_uniqueThreatCountHasBeenSet; }
  inline void SetUniqueThreatCount(int value) {
    m_uniqueThreatCountHasBeenSet = true;
    m_uniqueThreatCount = value;
  }
  inline MalwareScanDetails& WithUniqueThreatCount(int value) {
    SetUniqueThreatCount(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Threat> m_threats;
  bool m_threatsHasBeenSet = false;

  Aws::String m_scanId;
  bool m_scanIdHasBeenSet = false;

  MalwareProtectionScanType m_scanType{MalwareProtectionScanType::NOT_SET};
  bool m_scanTypeHasBeenSet = false;

  ScanCategory m_scanCategory{ScanCategory::NOT_SET};
  bool m_scanCategoryHasBeenSet = false;

  MalwareProtectionFindingsScanConfiguration m_scanConfiguration;
  bool m_scanConfigurationHasBeenSet = false;

  int m_uniqueThreatCount{0};
  bool m_uniqueThreatCountHasBeenSet = false;
};

}  // namespace Model
}  // namespace GuardDuty
}  // namespace Aws
